/**
 * Secret number guessing game
 * 
 * @author <a href='mailto:pandre.mudry&#64;hevs.ch'> Pierre-Andr Mudry</a>
 * @modified Pierre Roduit
 * @version 1.1
 */
public class SecretNumber {

	/**
	 * Ask the user a question with an integer answer
	 * 
	 * @param question The question to be asked
	 * @return The answer given (can only be an integer)
	 */
	public static int askUser(String question) {
		System.out.print(question);
		return Input.readInt();
	}

	/**
	 * Displays the initial menu of the application
	 */
	public static void displayMenu() {
		System.out.println(" ***********************************");
		System.out.println("* Secret Number - Can you find it ? *");
		System.out.println(" ***********************************");
		System.out.println("Choose difficulty level :\n");
		System.out.println("\t(1) Rookie level    (1 to 10)");
		System.out.println("\t(2) Easy level      (1 to 100)");
		System.out.println("\t(3) Normal level    (1 to 1000)");
		System.out.println("\t(4) Nightmare level (1 to 100000)\n");
	}

	/**
	 * Generates the secret number that has to be found in the range 1 to limit
	 * 
	 * @param limit The maximum number generated
	 * @return The generated random number
	 */
	public static int generateSecretNumber(int limit) {
		return (int) Math.floor(Math.random() * limit + 1);
	}

	/**
	 * Compute the number of iterations needed by the dichotomy algorithm
	 * 
	 * @param secretNumber The number to find
	 * @param upperLimit The biggest possible number
	 * @return The number of iterations that were taken
	 */
	public static int dichotomy_find(int secretNumber, int upperLimit) {
		int lowerLimit = 1;
		int middle = (upperLimit + lowerLimit) / 2;
		
		int nbIterations = 1;// We already have a number to test
		
		while (true) {
			// Number to search is bigger
			if (secretNumber > middle) {
				lowerLimit = middle;
				// There is a possibility that we got stuck with the
				// approximation (e.g. Limits are 2 and 3, and number to find is
				// 3, with the rounding down of the casting, the new middle
				// computed is always 2). So we round it up here
				middle = (int) Math.ceil((upperLimit + lowerLimit) / 2);
			}
			// Number to search is smaller
			else if (secretNumber < middle) {
				upperLimit = middle;
				// Here we round it down
				middle = (upperLimit + lowerLimit) / 2;
			}
			// We found the number
			else		
				return nbIterations;
					
			nbIterations++;
		}
	}

	public static void main(String args[]) {
		// Maximum number that can be generated
		int upperLimit = 0;

		// Number of tries
		int nTries = 0;

		while (true) {
			// First of all, let's draw the main menu
			displayMenu();

			int choice = askUser(">> What is your choice ? ");

			switch (choice) {
			case 1:
				upperLimit = 10;
				break;
			case 2:
				upperLimit = 100;
				break;
			case 3:
				upperLimit = 1000;
				break;
			case 4:
				upperLimit = 1000000;
				break;
			default:
				System.out.println("Choice not valid ! Retry ");
				continue;
			}

			int secretNumber = generateSecretNumber(upperLimit);

			// System.out.println("Number to be found " + secretNumber);

			// Do this until the user has made a valid guess
			while (true) {
				choice = askUser(">> Enter your guess (0 to quit): ");

				nTries++;

				// Handle the 'quit' choice
				if (choice == 0) {
					/* 
					 * Compute the number of iterations necessary with the 
					 * dichotomy search algorithm and display the resulting number					 
					 */
					System.out.println("The dichotomy algorithm took "
							+ dichotomy_find(secretNumber,upperLimit)
							+ " iterations to find this value");

					
					System.out.println("The correct answer was " + secretNumber);
					System.exit(-1);
				}

				// This is the correct choice
				if (choice == secretNumber) {
					// Displays different messages, depending on the number of tries
					if (nTries < 3) {
						System.out.println("\n**** Wonderful, you made it in less than 3 tries !");
					} else {
						System.out.println("\n!!!! Congratulations !!!! You made it in "+ nTries + " tries");
					}
					
					// 'Play again' logics
					System.out.print(">> Do you want to play again (y/n) ? ");
					char c = 0;

					/*
					 * We read characters from the console until has been
					 * entered.
					 */
					do {
						c = Input.readChar();

						// Optional error check
						if (c != 'y' && c != 'n') {
							System.out.print("Invalid choice, choose y or n :");
						}
					} while (c != 'y' && c != 'n');

					// The yes choice (user wants to play again)
					if (c == 'y') {
						// Resets the tries counter
						nTries = 0;

						// Break the infinite loop and go back to main menu
						break;
					} else {
						// Get out of here !
						System.out.println("Good bye !");
						System.exit(1);
					}
				}

				// Incorrect choice logics
				if (choice > secretNumber) {
					System.out.println("The number is smaller than " + choice);
				} else if (choice < secretNumber) {
					System.out.println("The number is bigger than " + choice);
				}
			}
		}
	}
}
